import numpy as np

# -------------------------------
# HDGL Lattice Core
# -------------------------------
class HDGLPage:
    """Represents a page of memory mapped to the lattice"""
    def __init__(self, name, initial_value=0.0):
        self.name = name
        self.energy = initial_value  # analog value
        self.last_access = 0

class HDGLChannel:
    """Analog channel for harmonic fusion"""
    def __init__(self, index):
        self.index = index
        self.energy = 0.0  # analog sum of assigned pages
        self.pages = []

    def add_page(self, page):
        self.pages.append(page)

    def propagate(self):
        # simple harmonic fusion: sum energies, decay slightly
        total = sum(p.energy for p in self.pages)
        self.energy = 0.9 * self.energy + 0.1 * total
        # feed back to pages (minimal, analog rounding)
        for p in self.pages:
            p.energy = 0.95 * p.energy + 0.05 * self.energy

class HDGLLattice:
    """The OS-invisible lattice"""
    def __init__(self, num_channels=32):
        self.channels = [HDGLChannel(i) for i in range(num_channels)]
        self.pages = {}

    def register_page(self, name, initial_value=0.0):
        page = HDGLPage(name, initial_value)
        self.pages[name] = page
        # assign page to a channel dynamically (round-robin or analog weight)
        ch_index = len(self.pages) % len(self.channels)
        self.channels[ch_index].add_page(page)

    def access_page(self, name, value):
        """Analog write operation"""
        page = self.pages[name]
        page.energy += value
        page.last_access += 1

    def propagate(self):
        """Propagate energy across all channels"""
        for ch in self.channels:
            ch.propagate()

# -------------------------------
# Debian Boot Simulation
# -------------------------------
def boot_debian_analog(lattice, kernel_pages, initramfs_pages, steps=64):
    print("Booting Debian via HDGL lattice...")

    # register all pages dynamically
    for i, kp in enumerate(kernel_pages):
        lattice.register_page(f"kernel_{i}", initial_value=np.random.rand())
    for i, ip in enumerate(initramfs_pages):
        lattice.register_page(f"initramfs_{i}", initial_value=np.random.rand()*0.5)

    # analog boot loop
    for t in range(steps):
        # random page accesses to simulate CPU reads/writes
        for _ in range(8):
            page_name = np.random.choice(list(lattice.pages.keys()))
            val = np.random.rand()
            lattice.access_page(page_name, val)

        lattice.propagate()  # harmonic fusion across lattice

        if t % 8 == 0:
            # print first 8 channel energies
            energies = [round(ch.energy, 3) for ch in lattice.channels[:8]]
            print(f"t={t}: Channels {energies}")

    print("Debian initramfs boot completed via HDGL.")
    return lattice

# -------------------------------
# Example Usage
# -------------------------------
kernel_pages = [f"vmlinuz_page_{i}" for i in range(64)]
initramfs_pages = [f"initramfs_page_{i}" for i in range(128)]

lattice = HDGLLattice(num_channels=32)
lattice = boot_debian_analog(lattice, kernel_pages, initramfs_pages)
